// ***********************************************************************
// Copyright © 2009-2013 Corporation for National Research Initiatives
// ("CNRI"). This software is made available under the terms of the
// CNRI License Agreement  that is located at hdl:1895.26/1012
// or http://hdl.handle.net/1895.26/1012.
// ***********************************************************************
//Developer: Robert R Tupelo-Schneck <schneck@cnri.reston.va.us>
//Developer: Kevin Hosford
// ***********************************************************************

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cr = Components.results;

function dump(msg) {
    Components.classes["@mozilla.org/consoleservice;1"]
        .getService(Components.interfaces.nsIConsoleService)
            .logStringMessage(msg);
}

function _alert(msg) {
    Cc["@mozilla.org/embedcomp/prompt-service;1"]
       .getService(Ci.nsIPromptService)
       .alert(null, 'Danger', msg);
}

function concatWithAmps(bss) {
    var len = bss.length;
    var amp = false;
    var res = "";
    for(var i = 0; i < len; i++) {
        if(bss[i] && bss[i+1].length > 0) {
            if(amp) res += "&";
            res += bss[i+1];
            amp = true;
        }
        i++;
    }
    return res;
}

// weird bug in Mac Firefox prevents use of JSObject in Java!
function optionsToString(options) {
    var res = "";
    for(att in options) {
        if(options[att]!=null) {
            res = res + att + ":" + encodeURIComponent(options[att]) + "\n";
        }
    }
    return res;
}

function removeEncodedLeadingWhitespace(s) {
    var qm = s.indexOf('?');
    var hm = s.indexOf('#');
    var end = s.length;
    if(qm >=0 && qm < end) end = qm;
    if(hm >=0 && hm < end) end = hm;
    var path = s.substring(0,end);
    var tail = s.substring(end);
    return encodeURIComponent(decodeURIComponent(path).replace(/^\s\s*/,'')).replace(/'/g, '%27').replace(/%2F/g,'/') + tail;
}

function looksLikeUri(spec) {
    var colon = spec.indexOf(':');
    if (colon < 0) return false;
    var slash = spec.indexOf('/');
    if (slash < 0) return true;
    if (slash < colon) return false;
    return true;
}

var prefService = null;
var paramString = "";
var showNotifications = true;
var persistNotifications = 0;
var notificationPersistTime = 5;
var keepHdlURIOnRedirect = true;
var options = null;
var prefObserver = {
    observe: function(subject, topic, data) {
        // dump("observe");
        showNotifications = getBoolPref("showNotifications",true);
        persistNotifications = getIntPref("persistNotifications",0);
        notificationPersistTime = getIntPref("notificationPersistTime",5);
        keepHdlURIOnRedirect = getBoolPref("keepHdlURIOnRedirect",true);
        paramString = concatWithAmps([getBoolPref("auth"),"auth",getBoolPref("cert"),"cert",getBoolPref("ignoreAliases"),"ignore_aliases",getBoolPref("noredirect"),"noredirect"]);
        
        var adminHandle = getCharPref("adminHandle");
        var displayType = getCharPref("displayType");
        var xslFile = getCharPref("xslFile");
        options = null;
        if(adminHandle!=null || displayType!=null || xslFile!=null) {
            options = {
                    adminHandle : adminHandle,
                    privateKeyFile : getCharPref("privateKeyFile"),
                    offset : getIntPref("offset"),
                    passphrase : getCharPref("passphrase"),
                    displayType : displayType,
                    xslFile : xslFile
            };
            if(options.offset==null) options.offset = getIntPref("publicKeyIndex"); // backward compatibility
            options = optionsToString(options);
        }
    }
}

var observingCategories = false;
var categorySinks = [];
var categoryObserver = {
    observe: function(subject, topic, data) {
        if(topic=="xpcom-shutdown") {
            var observerService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
            observerService.removeObserver(this,"xpcom-shutdown");
            observerService.removeObserver(this,"xpcom-category-entry-added");
            observerService.removeObserver(this,"xpcom-category-entry-removed");
            observerService.removeObserver(this,"xpcom-category-cleared");
            return;
        }
        if(data != "net-channel-event-sinks") return;
        // dump("observe " + subject + " " + topic + " " + data);
        var categoryManager = Cc["@mozilla.org/categorymanager;1"].getService(Ci.nsICategoryManager);
        var sinkContractIdEnumerator = categoryManager.enumerateCategory("net-channel-event-sinks");
        var newSinks = [];
        while(sinkContractIdEnumerator.hasMoreElements()) {
            try {       
                var s = sinkContractIdEnumerator.getNext().QueryInterface(Ci.nsISupportsCString).toString();
                // dump(s);
                newSinks[newSinks.length] = Cc[categoryManager.getCategoryEntry("net-channel-event-sinks",s)].getService(Ci.nsIChannelEventSink);
            }
            catch(err) {
                dump(err);
            }
        }
        categorySinks = newSinks;
    }
}

var getBoolPref = function(pref,def) {
    try {
        if (prefService) {
            if(prefService.prefHasUserValue(pref)) {
                return prefService.getBoolPref(pref);
            }
            else return def;
        }
    }
    catch(e) {
        return def;
    }
}

var getCharPref = function(pref,def) {
    var res = def;
    try {
        if (prefService) {
            if(prefService.prefHasUserValue(pref)) {
                res = prefService.getCharPref(pref);
            }
        }
    }
    catch(e) {
    }
    return res;
}
    
var getIntPref = function(pref,def) {
    var res = def;
    try {
        if (prefService) {
            if(prefService.prefHasUserValue(pref)) {
                res = prefService.getIntPref(pref).toString();
            }
        }
    }
    catch(e) {
    }
    if(res==null) return getCharPref(pref,def);
    return res;
}



function combineHandleParams(handle,params) {
    var hashIndex = handle.indexOf('#');
    var urlappend = null;
    if(hashIndex>=0) {
        urlappend = handle.substr(hashIndex+1);
        handle = handle.substr(0,hashIndex);
    }
    var questionIndex = handle.indexOf('?');
    var preParams = null;
    if(questionIndex>=0) {
        preParams = handle.substr(questionIndex+1);
        handle = handle.substr(0,questionIndex);
    }
    var postParams=null;
    var paramsList = null;
    if(preParams==null || preParams.length==0) postParams=params;
    else if(params==null || params.length==0) postParams=preParams;
    else {
        paramsList = [];
        
        var n;
        if(preParams!=null) {
            while((n = preParams.indexOf('&'))>=0) {
                paramsList.push(preParams.substr(0,n));
                preParams = preParams.substr(n+1);
            }
            if(preParams.length>0) paramsList.push(preParams);
        }
        
        if(params!=null) {
            while((n = params.indexOf('&'))>=0) {
                var thisParam = params.substr(0,n);
                // array indexOf is Firefox-specific
                if(paramsList.indexOf(thisParam)<0) paramsList.push(params.substr(0,n));
                params = params.substr(n+1);
            }
            if(params.length>0 && paramsList.indexOf(params)<0) paramsList.push(params);
        }
    }

    var res = handle;
    if(paramsList!=null && paramsList.length>0) {
        res = res + '?';
        var first = true;
        for(i = 0; i < paramsList.length; i++) {
            if(!first) res = res +'&';
            first = false;
            res = res + paramsList[i];
        }
    }
    if(postParams!=null && postParams.length>0) {
        res = res + '?';
        res = res + postParams;
    }
    if(urlappend!=null && urlappend.length>0) {
        res = res + '#';
        res = res + urlappend;
    }
    return res;
}


/*** HDLProtocolHandler ***/

function HDLProtocolHandler() {
    if(prefService==null) {
        prefService = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefService);
        prefService = prefService.getBranch("extensions.cnri-handle.").QueryInterface(Ci.nsIPrefBranch2);
        prefService.addObserver("", prefObserver, false);
        prefObserver.observe(null,null,null);
    }
    if(!observingCategories) {
        categoryObserver.observe(null,null,"net-channel-event-sinks");
        var observerService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
        observerService.addObserver(categoryObserver,"xpcom-shutdown",false);
        observerService.addObserver(categoryObserver,"xpcom-category-entry-added",false);
        observerService.addObserver(categoryObserver,"xpcom-category-entry-removed",false);
        observerService.addObserver(categoryObserver,"xpcom-category-cleared",false);
    }
}

HDLProtocolHandler.prototype.defaultPort = -1;
HDLProtocolHandler.prototype.protocolFlags = Ci.nsIProtocolHandler.URI_NORELATIVE | Ci.nsIProtocolHandler.URI_LOADABLE_BY_ANYONE;

HDLProtocolHandler.prototype.allowPort = function(port, scheme) {
    return false;
}

HDLProtocolHandler.prototype.newURI = function(spec, charset, baseURI) {
    var uri;
//    uri = Cc["@mozilla.org/network/simple-uri;1"].createInstance(Ci.nsIURI);
//    uri.spec = spec; // this can cause errors if we don't replace URI (in address bar) because we don't use relative uris
//                     // TODO consider keeping hdl/doi URI but if asked for a relative URI, return an HTTP etc URI instead here
    var good = looksLikeUri(spec);
    if (good) {
        try {
            uri = Cc["@mozilla.org/network/simple-uri;1"].createInstance(Ci.nsIURI);
            uri.spec = spec;
        } catch(e) {
            good = false;
        }
    } 
    if (!good) {
        var proxy = "hdl.handle.net";
        if(baseURI && baseURI.schemeIs("doi")) proxy = "dx.doi.org"; 
        uri = Components.classes["@mozilla.org/network/standard-url;1"].createInstance(Components.interfaces.nsIStandardURL);
        if (spec.indexOf("/")!=0) spec = "/" + spec;
        uri.init(Ci.nsIStandardURL.URLTYPE_AUTHORITY, 80, "http://" + proxy + spec, "UTF-8", null);
    }
    return uri;
}

HDLProtocolHandler.prototype.newChannel = function(URI) {
    var channel = new HdlChannel(URI);
    return channel.QueryInterface(Ci.nsIChannel);
}

HDLProtocolHandler.prototype.newChannel2 = function(URI, loadInfo) {
    var channel = new HdlChannel(URI, loadInfo);
    return channel.QueryInterface(Ci.nsIChannel);
}

HDLProtocolHandler.prototype.QueryInterface = function(iid) {
    if(!iid.equals(Ci.nsIProtocolHandler) && !iid.equals(Ci.nsISupports)) throw Cr.NS_ERROR_NO_INTERFACE;
    return this;
}

/**********************************************************************/
/*** HdlChannel ***/

var HdlChannel = function(URI, loadInfo) {
    this.wrappedJSObject = this;
    
    this.loadInfo = loadInfo;
    this.loadFlags = Ci.nsIRequest.LOAD_NORMAL;
    this.loadGroup = null;
    this.status = Cr.NS_OK;
    this.owner = null;
    this.URI = URI;
    this.originalURI = URI;
    this.notificationCallbacks = null;
    this.isPending = false;

    this.contentCharsetHint = null;
    this.contentLengthHint = null;
    this.contentTypeHint = null;

    this.cancelled = false;
    this.suspended = false;
    this.opened = false;
    this.listener = null;
    this.context = null;
    this.handler = null;
    
    this.proxy = null;
    this.proxyURI = null;

    this.proxyChannel = null;
    this.proxyChannelListener = null;
}

HdlChannel.prototype = {
    QueryInterface: function(iid){
        if (iid.equals(Ci.nsIChannel) || iid.equals(Ci.nsIRequest) || iid.equals(Ci.nsISupports))
            return this;
        // extras for HTTP redirection; TODO consider iid.equals(Ci.nsIHttpChannel)
        if (iid.equals(Ci.nsIUploadChannel) || iid.equals(Ci.nsIUploadChannel2))
            return this;
        throw Cr.NS_NOINTERFACE;
    },
   
//    cleanup: function() {
//        this.loadGroup = null;
//        this.notificationCallbacks = null;
//        this.contentCharsetHint = null;
//        this.contentLengthHint = null;
//        this.contentTypeHint = null;
//        this.isPending = false;
//        this.cancelled = false;
//        this.suspended = false;
//        this.opened = false;
//        this.listener = null;
//        this.context = null;
//        this.handler = null;
//        this.timerCallback = null;
//        this.timer = null;
//        
//        this.uploadCalled = 0;
//    },
  
    get name() {return this.URI},
    
    get contentCharset() {if(this.contentCharsetHint) return this.contentCharsetHint; return null;},
    set contentCharset(val) {this.contentCharsetHint = val},
    get contentLength() {if(this.contentLengthHint) return this.contentLengthHint; return -1},
    set contentLength(val) {this.contentLengthHint = val},
    get contentType() {if(this.contentTypeHint) return this.contentTypeHint; return null;},
    set contentType(val) {this.contentTypeHint = val},
    get securityInfo() {return null},
    
    get LOAD_NORMAL() {return Ci.nsIRequest.LOAD_NORMAL},
    get LOAD_BACKGROUND() {return Ci.nsIRequest.LOAD_BACKGROUND},
    get INHIBIT_CACHING() {return Ci.nsIRequest.INHIBIT_CACHING},
    get INHIBIT_PERSISTENT_CACHING() {return Ci.nsIRequest.INHIBIT_PERSISTENT_CACHING},
    get LOAD_BYPASS_CACHE() {return Ci.nsIRequest.LOAD_BYPASS_CACHE},
    get LOAD_FROM_CACHE() {return Ci.nsIRequest.LOAD_FROM_CACHE},
    get VALIDATE_ALWAYS() {return Ci.nsIRequest.VALIDATE_ALWAYS},
    get VALIDATE_NEVER() {return Ci.nsIRequest.VALIDATE_NEVER},
    get VALIDATE_ONCE_PER_SESSION() {return Ci.nsIRequest.VALIDATE_ONCE_PER_SESSION},
    
    get LOAD_DOCUMENT_URI() {return Ci.nsIChannel.LOAD_DOCUMENT_URI},
    get LOAD_RETARGETED_DOCUMENT_URI() {return Ci.nsIChannel.LOAD_RETARGETED_DOCUMENT_URI},
    get LOAD_REPLACE() {return Ci.nsIChannel.LOAD_REPLACE},
    get LOAD_INITIAL_DOCUMENT_URI() {return Ci.nsIChannel.LOAD_INITIAL_DOCUMENT_URI},
    get LOAD_TARGETED() {return Ci.nsIChannel.LOAD_TARGETED},

    isPending: function() {return this.isPending;},

    cancel: function(status) {
        if(this.cancelled) return;
        this.cancelled = true;
        this.status = status;
        this.isPending = false;
//        this.stopPolling();
//        if(this.handler!=null) {
//            this.handler.abort();
//        }
        if(this.proxyChannel) this.proxyChannel.cancel(status);
        //this.cleanup();
//        var self = this;
//        var callback = {
//            notify: function(timer) {
//                if(self.loadGroup) {
//                    self.loadGroup.QueryInterface(Ci.nsILoadGroup).removeRequest(self,null,Cr.NS_BINDING_REDIRECTED);
//                }
//                self.loadGroup = null;
//                self.cancelTimer = null;
//            }
//        }
//        this.cancelTimer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
//        this.cancelTimer.initWithCallback(callback,250,Ci.nsITimer.TYPE_ONE_SHOT);
    },
    resume: function() {
        this.suspended = false;
//        this.startPolling();
    },
    suspend: function() {
        this.suspended = true;
//        this.stopPolling();
    },

//    startPolling: function() {
//        if(this.timerCallback) {    
//            this.timer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
//            this.start = new Date().getTime();
//            this.timer.initWithCallback(this.timerCallback,250,Ci.nsITimer.TYPE_REPEATING_SLACK);
//        }
//    },
//    
//    stopPolling: function() {
//       if (this.timer) {
//           this.timer.cancel();
//           this.timer  = null;
//       }
//    },     
    
    open: function() {
        throw Cr.NS_ERROR_NOT_IMPLEMENTED;
    },

    setUploadStream: function(stream,contentType,contentLength) {
        this.uploadCalled = 1;
        this.uploadStream = stream;
        this.uploadContentType = contentType;
        this.uploadContentLength = contentLength;
    },
    
    explicitSetUploadStream: function(stream,contentType,contentLength,method,streamHasHeaders) {
        this.uploadCalled = 2;
        this.uploadStream = stream;
        this.uploadContentType = contentType;
        this.uploadContentLength = contentLength;
        this.uploadMethod = method;
        this.uploadStreamHasHeaders = streamHasHeaders;
    },
    
    asyncOpen: function(listener, context) {
        if(this.originalURI == this.proxyURI) this.originalURI = this.URI;
        if(this.opened) throw Cr.NS_ERROR_ALREADY_OPENED;
        var handle = null;
        if(this.proxy == null) {
            if (this.URI.scheme=='doi') this.proxy = 'http://dx.doi.org';
            else this.proxy = "http://hdl.handle.net";
        }
        var brand = this.URI.scheme;
        var doubleSlash;
        //if (this.URI.scheme=='doi') proxy = 'http://dx.doi.org/';
        if(this.URI.spec.indexOf("://") == 3) {
            handle = this.URI.spec.substr(6);
            doubleSlash = true;
        } 
        else {
            handle = this.URI.spec.substr(4);
            doubleSlash = false;
        }
        if(handle==null || handle.length==0 || handle.indexOf("?")==0 || handle.indexOf("#")==0) {
            if(this.loadFlags & this.LOAD_DOCUMENT_URI) {
                Cc["@mozilla.org/embedcomp/prompt-service;1"]
                    .getService(Ci.nsIPromptService)
                    .alert(null, 'Empty handle invalid.', 'Empty handle invalid.');
            }
            throw Cr.NS_BINDING_FAILED;
        }
        // remove leading whitespace
        handle = removeEncodedLeadingWhitespace(handle);
        
        this.opened = true;
        this.listener = listener;
        this.context = context;
        
        try {  
            this.isPending = true;

            var pes = null;
            var window = null;
            if(this.notificationCallbacks) {
                try { pes = this.notificationCallbacks.getInterface(Ci.nsIProgressEventSink); } catch(err) { pes=null; }
                if (pes) pes.onStatus(this,this.context,Ci.nsISocketTransport.STATUS_RESOLVING,handle);
                try { 
                    window = this.notificationCallbacks.getInterface(Ci.nsIDOMWindow); 
                    if (window==null) window = this.notificationCallbacks.getInterface(Ci.nsIWebProgress).DOMWindow; 
                } 
                catch (err) { window = null; }
            }

            if(this.loadGroup) {
                this.loadGroup.QueryInterface(Ci.nsILoadGroup).addRequest(this,null);
            
                if(this.loadGroup.notificationCallbacks) {
                    try { pes = this.loadGroup.notificationCallbacks.getInterface(Ci.nsIProgressEventSink); } catch(err) { pes=null; }
                    if (pes) pes.onStatus(this,this.context,Ci.nsISocketTransport.STATUS_RESOLVING,handle);
                    try {
                        if(window==null) window = this.loadGroup.notificationCallbacks.getInterface(Ci.nsIDOMWindow);
                        if(window==null) window = this.loadGroup.notificationCallbacks.getInterface(Ci.nsIWebProgress).DOMWindow;
                    }
                    catch (err) {
                        window = null;
                    }
                }
            }
            
            var chan = this;
            chan.handle = handle;
            chan.window = window;
            chan.brand = brand;
            chan.doubleSlash = doubleSlash;
            chan.paramString = paramString;
            chan.options = options;
            this.resolve();
//            if(this.handler==null) {
//                this.redirectToProxy(handle,paramString);
//            }
//            else {
//                this.timerCallback = new HandleTimerCallback(chan);
//                this.startPolling();
//            }
        } catch(err) {
            if (err.result != Cr.NS_BINDING_ABORTED) {
                throw err;
            }
        }
    },
        
    resolve: function() {
        var ioService = Components.classes["@mozilla.org/network/io-service;1"].getService(Components.interfaces.nsIIOService);
        var URL = this.proxy + "/" + combineHandleParams(this.handle,concatWithAmps([true,this.paramString,true,"forceProxy"]));
        var uri = ioService.newURI(URL,null,null);
        if (this.loadInfo) {
        	this.proxyChannel = ioService.newChannelFromURIWithLoadInfo(uri, this.loadInfo);
        } else {
        	this.proxyChannel = ioService.newChannelFromURI(uri);
        }
        var listener = new StreamListener(this);
        this.proxyChannel.notificationCallbacks = listener;
        this.proxyChannel.asyncOpen(listener,null);
    },
    
    showNotification: function(msg, chan, isRedirect) {
        var handle = chan.handle;
        var window = chan.window;
        chan.window = null;
        if(persistNotifications==0 && notificationPersistTime<=0) return;
        if(!showNotifications) return;
        if(!window) return;
        if(!window.top.document) return;
        var doc = window.top.document;
        try {
            var wm = Cc["@mozilla.org/appshell/window-mediator;1"].getService(Ci.nsIWindowMediator);
            var enumerator = wm.getEnumerator("navigator:browser");
            var tabbrowser = null;
            var currTab = null;
            var browserForDoc = null;
            while(tabbrowser==null && enumerator.hasMoreElements()) {
                window = enumerator.getNext();
                tabbrowser = window.getBrowser();
                browserForDoc = tabbrowser.getBrowserForDocument(doc);
                if(!browserForDoc) tabbrowser = null;
                else {
                    currTab = tabbrowser.selectedTab;
                    tabbrowser.selectedTab = tabbrowser.tabContainer.childNodes[tabbrowser.getBrowserIndexForDocument(doc)];
                }
            }
            if(tabbrowser==null) {
                return;  
            }
            var notificationbox = tabbrowser.getNotificationBox();
            var buttons = //[];
                [{
                    label:'Preferences...',
                    accessKey:null,
                    callback: function() {
                        var newWindow = window.openDialog("chrome://handle-extension/content/handle-preferences-standalone.xul");
                        newWindow.addEventListener("DOMContentLoaded",
                                function() {
                                    this.document.getElementById("handleTabList").selectedIndex = 1;
                                }, 
                                false);
                        return true;
                    }
                }];
            if(isRedirect) {
                buttons.unshift({
                    label:"Reload",
                    accessKey:null,
                    callback: function() {
                        browserForDoc.loadURI(chan.redirectURI);
                        return true;
                    }
                });
                buttons.unshift({
                    label:"Show Handle Values",
                    accessKey:null,
                    callback: function() {
                        browserForDoc.loadURI(chan.brand + (chan.doubleSlash ? "://" : ":") + handle + "?noredirect");
                        return true;
                    }
                });
            }
            notificationbox.appendNotification(msg, handle, chan.brand=="doi" ? 
                            "chrome://handle-extension/skin/images/doi-d.png" : 
                            "chrome://handle-extension/skin/images/h-triangle.png", 
                    notificationbox.PRIORITY_INFO_MEDIUM,
                    buttons);
            var notification = notificationbox.getNotificationWithValue(handle);
            notification.priority = notificationbox.PRIORITY_INFO_LOW;
            notification.persistence = persistNotifications;
            notification.timeout = new Date().getTime() + 1000 * notificationPersistTime;
            notificationbox.removeTransientNotifications();
            if(persistNotifications==0) {
                chan.notificationTimer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
                chan.notificationTimer.initWithCallback({
                        notify: function(timer) {
                            notification.close();
                            chan.notificationTimer = null;
                        }
                    },1000*notificationPersistTime,Ci.nsITimer.TYPE_ONE_SHOT);
            }
            tabbrowser.selectedTab = currTab;
        }
        catch(err) {
            dump("notification error (" + msg + "): " + err);
        }
        
    },
    
    redirectToChannel: function(newchan,noReplace) {
        if (!this.opened) throw Cr.NS_ERROR_UNEXPECTED;
        var chan = this;
        newchan.loadGroup = chan.loadGroup;
        newchan.notificationCallbacks = chan.notificationCallbacks;
        newchan.loadFlags = chan.loadFlags | chan.LOAD_REPLACE;
        if(noReplace && keepHdlURIOnRedirect) {
            try {
                var bag = newchan.QueryInterface(Ci.nsIWritablePropertyBag2);
                if(bag) {
                    bag.setPropertyAsInterface("baseURI",newchan.URI);
                    newchan.loadFlags = chan.loadFlags;
                }
            }
            catch(e) {}
        }
        
        if(this.uploadCalled>0) {
            var uploadChannel = null;
            try { uploadChannel = newchan.QueryInterface(Ci.nsIUploadChannel); } catch(err) {}
            var uploadChannel2 = null;
            try { uploadChannel2 = newchan.QueryInterface(Ci.nsIUploadChannel2); } catch(err) {}
            if(uploadCalled==1) {
                if(uploadChannel) {
                    uploadChannel.setUploadStream(this.uploadStream,this.uploadContentType,this.uploadContentLength);
                }
                else if(uploadChannel2) {             
                    // hopefully unreachable, but just in case
                    uploadChannel2.explicitSetUploadStream(this.uploadStream,this.uploadContentType,this.uploadContentLength,"",false);
                }
            }
            else if(uploadCalled==2) {
                if(uploadChannel2) {             
                    uploadChannel2.explicitSetUploadStream(this.uploadStream,this.uploadContentType,this.uploadContentLength,this.uploadMethod,this.uploadStreamHasHeaders);
                }
                else if(uploadChannel) {
                    if (this.uploadStreamHasHeaders) {
                        uploadChannel.setUploadStream(this.uploadStream,"",-1);
                    }
                    else {
                        uploadChannel.setUploadStream(this.uploadStream,this.uploadContentType,this.uploadContentLength);
                    }
                }
            }
        }
        
        var sinks = [];
        var globalChannelEventSink = Cc["@mozilla.org/netwerk/global-channel-event-sink;1"];
        if (globalChannelEventSink) {
        	sinks = [ globalChannelEventSink.getService().QueryInterface(Ci.nsIChannelEventSink) ];
        }
        var currentCategorySinks = categorySinks;
        sinks = sinks.concat(currentCategorySinks);
        var ces = null;
        if(newchan.notificationCallbacks) {
            try { ces = newchan.notificationCallbacks.getInterface(Ci.nsIChannelEventSink); } catch(err) { ces=null; }
            if (ces) sinks[sinks.length] = ces;
        }
        if(ces==null && newchan.loadGroup && newchan.loadGroup.notificationCallbacks) {
            try { ces = newchan.loadGroup.notificationCallbacks.getInterface(Ci.nsIChannelEventSink); } catch(err) { ces=null; }
            if (ces) sinks[sinks.length] = ces;
        }

        
        var sinkCount = 0;
        var bad = false;
        var onRedirectVerifyCallback = {
            onRedirectVerifyCallback: function(res) {
                sinkCount++;
                if(res & 0x80000000) {
                    bad = true;
                }
                if(sinkCount==sinks.length) {
                    if(bad) {
                        //chan.cancel(Cr.NS_BINDING_FAILED);
                        if(chan.listener) {
                            chan.contentTypeHint = "text/plain";
                            chan.listener.QueryInterface(Ci.nsIRequestObserver).onStartRequest(chan,chan.context);
                            if(!chan.cancelled) {
                                var s = "CNRI handle extension redirect aborted.";
                                var len = s.length;
                                var is = Cc["@mozilla.org/io/string-input-stream;1"].createInstance(Components.interfaces.nsIStringInputStream);
                                is.setData(s,len);
                                chan.listener.QueryInterface(Ci.nsIStreamListener).onDataAvailable(chan,chan.context,is,0,len);
                            }
                            chan.listener.QueryInterface(Ci.nsIRequestObserver).onStopRequest(chan,chan.context,Cr.NS_BINDING_FAILED);
                        }
                        if(chan.loadGroup) {
                            chan.loadGroup.QueryInterface(Ci.nsILoadGroup).removeRequest(chan,null,Cr.NS_BINDING_FAILED);
                        }    
                        chan.cancel(Cr.NS_BINDING_FAILED);
                    }
                    else {
                        newchan.originalURI = chan.originalURI;
                        if (chan.contentCharsetHint) newchan.contentCharset = chan.contentCharsetHint;
                        if (chan.contentLengthHint) newchan.contentLength = chan.contentLengthHint;
                        if (chan.contentTypeHint) newchan.contentType = chan.contentTypeHint;        
                        if(chan.loadGroup) {
                            chan.loadGroup.QueryInterface(Ci.nsILoadGroup).removeRequest(chan,null,Cr.NS_BINDING_REDIRECTED);
                        }    
                        newchan.asyncOpen(chan.listener,chan.context);
                        chan.cancel(Cr.NS_BINDING_REDIRECTED);
                    }
                    chan.listener = null;
                    chan.context = null;
                    chan.opened = false;
                }
            }
        }

        for(var i = 0; i < sinks.length; i++) {
            try {
                if(!sinks[i].asyncOnChannelRedirect) {
                    sinks[i].onChannelRedirect(chan,newchan,sinks[i].REDIRECT_TEMPORARY);
                    onRedirectVerifyCallback.onRedirectVerifyCallback(Cr.NS_OK);
                }
                else {
                    sinks[i].asyncOnChannelRedirect(chan,newchan,sinks[i].REDIRECT_TEMPORARY,onRedirectVerifyCallback);
                }
            }
            catch(e) {
                onRedirectVerifyCallback.onRedirectVerifyCallback(e);
            }
        }
    },
    
    redirectToURL: function(URL) {
        var ioServ = Cc["@mozilla.org/network/io-service;1"].getService();
        ioServ = ioServ.QueryInterface(Ci.nsIIOService);
        var newchan;
        var uri = ioServ.newURI(URL,null,null);
        if (this.loadInfo) {
        	newchan = ioService.newChannelFromURIWithLoadInfo(uri, loadInfo);
        } else {
        	newchan = ioService.newChannelFromURI(uri);
        }
        this.redirectToChannel(newchan,true);
    },
    
    redirectToProxy: function(handle,paramString,noReplace) {
        var URL = this.proxy + "/" + combineHandleParams(handle,concatWithAmps([true,paramString,true,"forceProxy"]));
        var ioServ = Cc["@mozilla.org/network/io-service;1"].getService();
        ioServ = ioServ.QueryInterface(Ci.nsIIOService);
        var uri = ioServ.newURI(URL,null,null);
        if (this.loadInfo) {
        	newchan = ioService.newChannelFromURIWithLoadInfo(uri, loadInfo);
        } else {
        	newchan = ioService.newChannelFromURI(uri);
        }
        this.redirectToChannel(newchan,noReplace);
    },
    
    writeAll: function(buf) {
        var converter = Cc["@mozilla.org/intl/scriptableunicodeconverter"].createInstance(Components.interfaces.nsIScriptableUnicodeConverter);
        converter.charset = "UTF-8";
        var inputStreamChannel = Cc["@mozilla.org/network/input-stream-channel;1"].createInstance(Ci.nsIInputStreamChannel);
        inputStreamChannel.setURI(this.URI);
        inputStreamChannel.contentStream = converter.convertToInputStream(buf); 
        this.redirectToChannel(inputStreamChannel.QueryInterface(Ci.nsIChannel),false);
    }
}
    
function StreamListener(chan) {
    this.chan = chan;
    chan.contentCharset = "UTF-8";
    this.data = "";
    this.scriptableInputStream =
        Components.classes["@mozilla.org/scriptableinputstream;1"]
          .createInstance(Components.interfaces.nsIScriptableInputStream);
    this.utf8Converter = Components.classes["@mozilla.org/intl/utf8converterservice;1"].
        getService(Components.interfaces.nsIUTF8ConverterService);
}

StreamListener.prototype.onStartRequest = function(aRequest,aContext) {
    this.data = "";
};

StreamListener.prototype.onDataAvailable = function(aRequest,aContext,aStream,aSourceOffset,aLength) {
    this.scriptableInputStream.init(aStream);
    var str = this.scriptableInputStream.readBytes(aLength);
    this.data += this.utf8Converter.convertStringToUTF8 (str, "UTF-8", false); 
};

StreamListener.prototype.onStopRequest = function(aRequest,aContext,aStatus) {
    var chan = this.chan;
    if(chan.redirectURI) return;
    chan.isPending = false;
    if(Components.isSuccessCode(aStatus)) {
        chan.writeAll(this.data);
    } else {
        chan.showNotification((chan.brand=="doi"?"DOI name ":"Handle ") + decodeURIComponent(chan.handle) + " lookup failed.  Trying proxy " + chan.proxy, chan, false);
        chan.redirectToProxy(chan.handle,chan.paramString);
    }
};

StreamListener.prototype.asyncOnChannelRedirect = function(aOldChannel,aNewChannel,aFlags,aCallback) {
    var chan = this.chan;
    chan.redirectURI = aNewChannel.URI.spec;
    chan.showNotification((chan.brand=="doi"?"DOI name ":"Handle ") + decodeURIComponent(chan.handle) + " was redirected to " + chan.redirectURI, chan, true);
    chan.redirectToChannel(aNewChannel,true);
    aCallback.onRedirectVerifyCallback(Cr.NS_OK);
    aOldChannel.cancel(Cr.NS_BINDING_REDIRECTED);
};

StreamListener.prototype.getInterface = function (aIID) {
  try {
    return this.QueryInterface(aIID);
  } catch (e) {
    throw Components.results.NS_NOINTERFACE;
  }
};

// nsIProgressEventSink (not implementing will cause annoying exceptions)
StreamListener.prototype.onProgress = function (aRequest, aContext, aProgress, aProgressMax) { };
StreamListener.prototype.onStatus = function (aRequest, aContext, aStatus, aStatusArg) { };
// nsIHttpEventSink (not implementing will cause annoying exceptions)
StreamListener.prototype.onRedirect = function (aOldChannel, aNewChannel) { };

// we are faking an XPCOM interface, so we need to implement QI
StreamListener.prototype.QueryInterface = function(aIID) {
  if (aIID.equals(Components.interfaces.nsISupports) ||
      aIID.equals(Components.interfaces.nsIInterfaceRequestor) ||
      aIID.equals(Components.interfaces.nsIChannelEventSink) ||
      aIID.equals(Components.interfaces.nsIProgressEventSink) ||
      aIID.equals(Components.interfaces.nsIHttpEventSink) ||
      aIID.equals(Components.interfaces.nsIStreamListener))
    return this;

  throw Components.results.NS_NOINTERFACE;
};

/**********************************************************************/
/*** HDLProtocolHandlerFactory ***/

var HDLProtocolHandlerFactory = {
    createInstance: function(outer, iid) {
        if(outer != null) throw Cr.NS_ERROR_NO_AGGREGATION;
        return (new HDLProtocolHandler()).QueryInterface(iid);
    },
    
    QueryInterface: function(iid) {
        if(!iid.equals(Ci.nsIFactory) && !iid.equals(Ci.nsISupports)) throw Cr.NS_ERROR_NO_INTERFACE;
        return this;
    }
}

/**********************************************************************/
/*** HDLModule ***/

var HDLModule = new Object();

HDLModule.registerSelf = function(compMgr, fileSpec, location, type) {
    compMgr = compMgr.QueryInterface(Ci.nsIComponentRegistrar);
    compMgr.registerFactoryLocation(Components.ID("{e6551680-a45e-11dd-89ac-0002a5d5c51b}"),
                                    "CNRI Handle System protocol handler",
                                    "@mozilla.org/network/protocol;1?name=hdl",
                                    fileSpec, location, type);
    compMgr.registerFactoryLocation(Components.ID("{94c29c00-aab4-11dd-b276-0002a5d5c51b}"),
                                    "DOI protocol handler",
                                    "@mozilla.org/network/protocol;1?name=doi",
                                    fileSpec, location, type);
    if(false) compMgr.registerFactoryLocation(Components.ID("{4ee3b7a0-fbc1-11de-8a39-0800200c9a66}"),
                                    "get protocol handler",
                                    "@mozilla.org/network/protocol;1?name=get",
                                    fileSpec, location, type);
}

HDLModule.unregisterSelf = function(compMgr, location, loaderStr) {
    compMgr = compMgr.QueryInterface(Ci.nsIComponentRegistrar);
    compMgr.unregisterFactoryLocation(Components.ID("{e6551680-a45e-11dd-89ac-0002a5d5c51b}"), location);
    compMgr.unregisterFactoryLocation(Components.ID("{94c29c00-aab4-11dd-b276-0002a5d5c51b}"), location);
    if(false) compMgr.unregisterFactoryLocation(Components.ID("{4ee3b7a0-fbc1-11de-8a39-0800200c9a66}"), location);
}

HDLModule.getClassObject = function(compMgr, cid, iid) {
    if(!iid.equals(Ci.nsIFactory)) throw Cr.NS_ERROR_NOT_IMPLEMENTED;
    if(cid.equals(Components.ID("{94c29c00-aab4-11dd-b276-0002a5d5c51b}")) || cid.equals(Components.ID("{e6551680-a45e-11dd-89ac-0002a5d5c51b}"))) return HDLProtocolHandlerFactory;
    if(false && cid.equals(Components.ID("{4ee3b7a0-fbc1-11de-8a39-0800200c9a66}"))) return HDLProtocolHandlerFactory;
    throw Cr.NS_ERROR_NO_INTERFACE;
}

HDLModule.canUnload = function(compMgr) {
    return true;
}

function NSGetModule(compMgr, fileSpec) {
    return HDLModule;
}

function NSGetFactory(cid) {
    return HDLProtocolHandlerFactory;
}
